import socket
import threading
import time
import tkinter as tk


class QueryError(Exception):
    pass


class GUI(tk.Tk):
    def __init__(self):
        super().__init__()
        self.title("ark-switch | by clemens")
        self.geometry("600x600+660+240")

        # status label
        self.label1 = tk.Label(self, text="", font=("Noto Mono", 40))
        self.label1.pack(pady=30)

        # image files
        self.on = tk.PhotoImage(file="./media/lever_on.png")
        self.off = tk.PhotoImage(file="./media/lever_off.png")

        # switch
        self.switch = tk.Button(self, bd=0, command=self.switch_switch, state=tk.DISABLED)
        self.switch.pack(pady=50)

        # log label
        self.log_label = tk.Label(self, text="", font=("Consolas", 16))
        self.log_label.pack()

        # popup
        self.client = None
        self.current_img = None
        self.pass_popup()

    def pass_popup(self):
        top = tk.Toplevel(self)
        top.attributes("-topmost", True)
        top.geometry("200x100+860+490")
        top.title("")

        label = tk.Label(top, text="please enter server password")
        label.pack()

        entry = tk.Entry(top, width=25)
        entry.pack()

        def get_password():
            # creating server client
            self.client = Client(entry.get().encode())
            time.sleep(1)
            try:
                self.state = not self.client.get_state()
            except QueryError:
                print("wrong password")
                self.client.close()
                return

            self.current_img = tk.PhotoImage()

            # enabling button
            self.switch.config(state=tk.NORMAL)

            # setting the state correctly
            self.switch_switch(coms=False)

            self.attributes("-topmost", True)
            top.destroy()

        submit_btn = tk.Button(top, text="submit", command=get_password)
        submit_btn.pack()

    # switching the switch
    def switch_switch(self, coms=True):
        self.state = not self.state
        if self.state:
            self.current_img = self.on
            self.label1.config(text="Server is on!", bg="green")
            if coms:
                try:
                    self.log_label.config(text=self.client.set_state(1))
                except TimeoutError:
                    self.log_label.config(text="connection timed out")
                    time.sleep(5)
                    self.destroy()
                    return
        else:
            self.current_img = self.off
            self.label1.config(text="Server is off!", bg="red")
            if coms:
                try:
                    self.log_label.config(text=self.client.set_state(0))
                except TimeoutError:
                    self.log_label.config(text="connection timed out")
                    time.sleep(5)
                    self.destroy()
                    return

        self.switch.config(image=self.current_img)

    def destroy(self) -> None:
        if self.client:
            self.client.close()
            print("closed connection.")
        super().destroy()


class Client:
    def __init__(self, password: bytes):
        # host variables
        self.sock = socket.socket()
        self.port: int = 6969
        self.host: str = "rggampel.zapto.org"

        # initial connect
        self.sock.connect((self.host, self.port))
        self.sock.settimeout(30)

        # authorize connection
        # self.sock.send(input("input server password\n> ").encode())
        self.sock.send(password)
        # getting response to authentication
        print(self.sock.recv(1024).decode())

        # starting keep alive pings
        self.alive = True  # variable so we stop pinging if the connection is dead
        self.keep_alive()

    # sending keep alive pings
    def keep_alive(self):
        def keep_alive_thread():
            while 1:
                if self.alive:
                    self.sock.send(b"k")
                    time.sleep(10)
                else:
                    break
        t = threading.Thread(daemon=True, target=keep_alive_thread)
        t.start()

    # method for getting the state of the ark server
    def get_state(self) -> bool:
        # sending request
        self.sock.send(b"?")

        # getting response
        try:
            response_header, response = self.sock.recv(512).decode().split(";")
        except ValueError:
            raise QueryError("invalid response.")

        # if everything went ok we return respective values
        if int(response_header) == 0:
            return bool(int(response))
        # raise error if error happened
        else:
            raise QueryError(response_header, response)

    # method for setting ark server state
    def set_state(self, state: int) -> str:
        # sending request
        self.sock.send(("!" + str(state)).encode())

        # handling response
        response = self.sock.recv(4056).decode().split(";")
        if header := int(response[0]) == 0:
            return response[1]
        else:
            raise QueryError(header, response)

    # closing connection cleanly
    def close(self):
        self.sock.close()
        self.alive = False

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.close()


if __name__ == '__main__':
    gui = GUI()
    gui.mainloop()
